%% SIR parameter studies

% Runs an SIR experiment to determine how maximum infected and final at 
% risk fractions depend on a parameter value
%
% Uses sir_sim.m, 2020/10/26 version
%
% User specifies a list of values for one of the key parameters:
%   R0 is the basic reproductive number
%   T is the mean infectious duration
%   I0 is the initial infectious fraction
%   V is the initial immune fraction
%
% The program is designed so that only a few lines need to be modified to
% make a new experiment (see '%%%' comments)
%   lines 52-54 define the independent variable values
%   line 57 identifies the x axis label for the graph
%   line 85 links the independent variable name and values
%
% Output figure:
%   top left panel: max infected fraction
%   top right panel: day of max infected fraction
%   bottom left panel: final fraction susceptibles
%   bottom right panel: day when end condition is reached
%
% Y contains one row for each value of the independent variable
% Columns are
%   1. maximum infected fraction
%   2. day of max new infections
%   3. final susceptible fraction
%   4. days until I<target
%
% by Glenn Ledder
% written 2020/06/21
% revised 2020/10/26
%
% direct comments to gledder@unl.edu

%% DEFAULT SCENARIO DATA

R0 = 5;
T = 5;
I0 = 0.001;
V = 0;

%% INDEPENDENT VARIABLE DATA

%%% This section needs to be modified for each experiment.

%%% first and last are the min and max values for the independent variable
%%% N is the number of points (not subdivisions)
first = 0;
last = 5;
N = 51;

%%% xname is the name for the x axis label
xname = 'R0';

%% COMMON DATA

target = 0.001;

%% INITIALIZATION

opengl hardwarebasic;
clf
for k=1:4
    subplot(2,2,k)
    hold on
    xlabel(xname)
end

% xvals holds whatever values are being used for the independent variable
xvals = linspace(first,last,N);

% rows are for each n
Y = zeros(N,4);

%% COMPUTATION

for n=1:N
    
    %%% The left side of this statement needs to be the independent
    %%% variable for the experiment.
    R0 = xvals(n);

    gamma = 1/T;
    beta = R0*gamma;
    [S,I,~] = sir_sim(beta,gamma,I0,V,target);

    [Y(n,1),Y(n,2)] = max(I);
    Y(n,2) = Y(n,2)-1;

    Y(n,3) = (S(1)-S(end))/S(1);
    Y(n,4) = length(I)-1;
end
    
%% OUTPUT

for k=1:4
    subplot(2,2,k)
    plot(xvals,Y(:,k),'LineWidth',1.7)
end

subplot(2,2,1)
ylabel('max fraction infected')
subplot(2,2,2)
ylabel('days for max')
subplot(2,2,3)
ylabel('final fraction susceptible')
subplot(2,2,4)
ylabel('days to end condition')

